# File Structure & Explanation

## 📁 Configuration Files (`/config`)

### `db.php`
- **Purpose**: Database connection and helper functions
- **Key Features**:
  - PDO connection with error handling
  - Prepared statements support
  - Input sanitization function
  - Secure token generation
- **Security**: Uses PDO with prepared statements to prevent SQL injection

### `auth.php`
- **Purpose**: Authentication and session management
- **Key Functions**:
  - `isAdminLoggedIn()`: Check if admin is authenticated
  - `requireAdminLogin()`: Protect admin pages
  - `redirectIfLoggedIn()`: Redirect if already logged in
- **Security**: Session-based authentication

### `qr.php`
- **Purpose**: QR code generation helper
- **Features**:
  - Uses external QR API service (qrserver.com)
  - Fallback for offline generation (placeholder)
- **Note**: For production, consider installing a PHP QR library

---

## 🔐 Admin Files (`/admin`)

### `login.php`
- **Purpose**: Admin login page
- **Features**:
  - Email/password authentication
  - AJAX form submission (no page reload)
  - Alert notifications
  - Session protection (redirects if already logged in)
- **Backend**: Calls `/api/login.php`

### `dashboard.php`
- **Purpose**: Admin dashboard for managing employees
- **Features**:
  - Create new employees
  - Display all employees with QR codes
  - Download QR codes as images
  - Logout button
- **Security**: Protected by `requireAdminLogin()`
- **Backend**: 
  - Fetches employees from database
  - Calls `/api/create_employee.php` for new employees

### `logout.php`
- **Purpose**: Destroy session and logout
- **Action**: Destroys session and redirects to login

---

## 🔌 API Endpoints (`/api`)

### `login.php`
- **Method**: POST
- **Input**: `{ email, password }`
- **Output**: JSON response
- **Features**:
  - Validates email format
  - Verifies password hash
  - Creates session on success
- **Security**: Password hashing with `password_verify()`

### `create_employee.php`
- **Method**: POST
- **Input**: `{ name, department }`
- **Output**: JSON with created employee data
- **Features**:
  - Generates unique secure token
  - Validates input
  - Returns employee with QR token
- **Security**: Requires admin authentication

### `attendance.php`
- **Method**: POST
- **Input**: `{ token, action }` (action: 'check-in' or 'check-out')
- **Output**: JSON response
- **Features**:
  - Validates token
  - Enforces attendance rules:
    - One check-in per day
    - Check-out only after check-in
    - Prevents duplicates
  - Uses server time (not browser time)
- **Security**: Token-based (no employee ID exposure)

---

## 👤 Employee Files

### `attendance.php` (root)
- **Purpose**: Employee attendance page (public via QR)
- **Features**:
  - Validates QR token from URL
  - Displays employee name and department
  - Check-in/Check-out form
  - AJAX submission
- **Access**: Only via valid QR token
- **Security**: Token validation, no session required

---

## 🗄️ Database

### `database.sql`
- **Purpose**: Database schema and initial data
- **Tables**:
  - `admins`: Admin accounts with hashed passwords
  - `employees`: Employee data with unique QR tokens
  - `attendance`: Daily attendance records
- **Indexes**: Optimized for performance
- **Default Admin**: Includes default admin account (change password!)

---

## 🎨 Frontend Assets

### CSS Files (`/css`)
- `global.css`: Base styles
- `login.css`: Login page styles
- `dashboard.css`: Dashboard styles
- `attendence.css`: Attendance page styles
- `alert.css`: Alert/notification styles

### Assets (`/assets`)
- SVG images for UI elements

---

## 🔧 Configuration Files

### `.htaccess`
- **Purpose**: Apache configuration for shared hosting
- **Features**:
  - Protects config files
  - Security headers
  - Error handling
  - Session security settings

---

## 🔄 Data Flow

### Admin Login Flow
1. User submits form → `admin/login.php`
2. JavaScript sends AJAX → `/api/login.php`
3. API validates credentials → Creates session
4. Redirects to → `admin/dashboard.php`

### Employee Creation Flow
1. Admin submits form → `admin/dashboard.php`
2. JavaScript sends AJAX → `/api/create_employee.php`
3. API creates employee → Generates token
4. Returns employee data → Dashboard displays QR code

### Attendance Flow
1. Employee scans QR → Opens `attendance.php?token=XXX`
2. Page validates token → Displays employee info
3. Employee selects action → Submits form
4. JavaScript sends AJAX → `/api/attendance.php`
5. API validates rules → Records attendance
6. Returns success/error → User sees notification

---

## 🔒 Security Features

1. **Password Hashing**: Uses `password_hash()` and `password_verify()`
2. **SQL Injection Prevention**: PDO prepared statements only
3. **Input Sanitization**: All inputs sanitized with `sanitize()`
4. **Session Security**: HTTP-only cookies, secure sessions
5. **Token-Based Access**: Employees access via tokens, not IDs
6. **Server-Side Validation**: All rules enforced on server
7. **Protected Files**: `.htaccess` blocks config file access

---

## 📝 Notes

- **No Frameworks**: Pure PHP, HTML, CSS, JavaScript
- **Shared Hosting Ready**: Works on cPanel and similar hosts
- **Production Ready**: Security best practices implemented
- **QR Code Service**: Uses external API (can be replaced with library)
- **Base URL**: Auto-detected, but should be set manually for production

---

## 🚀 Deployment Checklist

- [ ] Import `database.sql`
- [ ] Update `config/db.php` credentials
- [ ] Change default admin password
- [ ] Set base URL in `dashboard.php`
- [ ] Upload all files to server
- [ ] Set proper file permissions
- [ ] Test login functionality
- [ ] Test employee creation
- [ ] Test QR code scanning
- [ ] Test attendance recording
- [ ] Enable HTTPS (recommended)
